from django.shortcuts import render, redirect
from django.views.decorators.http import require_POST
from django.contrib import messages
from django.http import JsonResponse


def _get_cart(request):
    """
    Obtiene el carrito desde la sesión.
    Estructura esperada:
    {
        "SKU001": {"name": "Producto X", "price": 1000, "qty": 2},
        ...
    }
    """
    return request.session.get("cart", {})


def _save_cart(request, cart):
    """
    Guarda el carrito en la sesión y marca la sesión como modificada.
    """
    request.session["cart"] = cart
    request.session.modified = True


@require_POST
def cart_add(request):
    """
    Agrega un producto al carrito.
    Soporta:
      - Petición normal (redirect)
      - Petición AJAX (fetch) → responde JSON
    """
    sku = request.POST.get("sku")
    name = request.POST.get("name")
    price = request.POST.get("price")

    if not (sku and name and price):
        if request.headers.get("X-Requested-With") == "XMLHttpRequest":
            return JsonResponse({"success": False, "error": "Datos incompletos."}, status=400)
        messages.error(request, "No se pudo agregar el producto al carrito.")
        return redirect("products:product_list")

    try:
        price = float(price)
    except ValueError:
        if request.headers.get("X-Requested-With") == "XMLHttpRequest":
            return JsonResponse({"success": False, "error": "Precio inválido."}, status=400)
        messages.error(request, "Precio inválido.")
        return redirect("products:product_list")

    cart = _get_cart(request)

    if sku in cart:
        cart[sku]["qty"] += 1
    else:
        cart[sku] = {
            "name": name,
            "price": price,
            "qty": 1,
        }

    _save_cart(request, cart)

    # calcular cantidad total para el badge
    total_qty = sum(int(item.get("qty", 0)) for item in cart.values())

    # Si es AJAX → devolvemos JSON
    if request.headers.get("X-Requested-With") == "XMLHttpRequest":
        return JsonResponse(
            {
                "success": True,
                "cart_item_count": total_qty,
                "message": f"Se agregó {name} al carrito.",
            }
        )

    # Si no es AJAX → comportamiento normal
    messages.success(request, f"Se agregó {name} al carrito.")
    return redirect("products:product_list")


def cart_detail(request):
    """
    Muestra el contenido del carrito.
    """
    cart = _get_cart(request)

    items = []
    total = 0

    for sku, item in cart.items():
        try:
            price = float(item.get("price", 0))
        except (TypeError, ValueError):
            price = 0

        try:
            qty = int(item.get("qty", 1))
        except (TypeError, ValueError):
            qty = 1

        subtotal = price * qty
        total += subtotal

        items.append(
            {
                "sku": sku,
                "name": item.get("name"),
                "price": price,
                "qty": qty,
                "subtotal": subtotal,
            }
        )

    context = {
        "items": items,
        "total": total,
    }
    return render(request, "cart/cart_detail.html", context)


def cart_remove(request, sku):
    """
    Elimina un producto del carrito por su SKU.
    """
    cart = _get_cart(request)
    if sku in cart:
        name = cart[sku].get("name", "Producto")
        del cart[sku]
        _save_cart(request, cart)
        messages.info(request, f"Se eliminó {name} del carrito.")
    return redirect("cart:detail")


def cart_clear(request):
    """
    Vacía completamente el carrito.
    """
    _save_cart(request, {})
    messages.info(request, "Se vació el carrito.")
    return redirect("cart:detail")
